/*
 * Decompiled with CFR 0.152.
 */
package biz.papercut.pcng.common;

import biz.papercut.pcng.util.Formatter;
import biz.papercut.pcng.util.Messages;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ChargeRate
implements Serializable {
    private static final Logger logger = LoggerFactory.getLogger(ChargeRate.class);
    private static final long serialVersionUID = 1L;
    private static final double DEFAULT_RATE = 100.0;
    private static final String DEFAULT_RATE_FORMATTED = "+0.0%";
    private static final char VALUE_SEPERATOR = ':';
    private static final char LIST_SEPERATOR = ',';
    private final String _description;
    private final double _amount;
    private final String _formattedAmount;
    private final ChargeRateType _type;

    public ChargeRate(String description, double amount, String formattedAmount, ChargeRateType type) {
        if (type == null) {
            throw new NullPointerException("Null charge rate type");
        }
        this._amount = amount;
        this._description = StringUtils.trimToEmpty((String)description);
        this._type = type;
        this._formattedAmount = StringUtils.trimToEmpty((String)formattedAmount);
        if (this._type != ChargeRateType.Percentage && this._formattedAmount.length() == 0) {
            throw new IllegalArgumentException("empty formatted amount for currency charge rate");
        }
        if (this._description.length() == 0) {
            throw new IllegalArgumentException("charge rate description required");
        }
        if (this._type == ChargeRateType.Percentage && amount < 0.0) {
            throw new IllegalArgumentException("charge rate % must not be negative");
        }
    }

    public static ChargeRate getDefaultChargeRate() {
        return new ChargeRate(Messages.getString(ChargeRate.class, "ChargeRate.DEFAULT", new String[0]), 100.0, DEFAULT_RATE_FORMATTED, ChargeRateType.Percentage);
    }

    public boolean doesRateAdjustCost() {
        switch (this._type) {
            case Percentage: {
                double tolerance = 1.0E-4;
                return this.getAmount() > 100.0001 || this.getAmount() < 99.9999;
            }
            case PerJob: 
            case PerPage: 
            case PerCopy: {
                return this.getAmount() != 0.0;
            }
        }
        return true;
    }

    public String getDescription() {
        return this._description;
    }

    public double getAmount() {
        return this._amount;
    }

    public String getFormattedAmount() {
        return this._formattedAmount;
    }

    public ChargeRateType getType() {
        return this._type;
    }

    public double adjustCost(double cost, int totalPages, int copies) {
        double result;
        switch (this._type) {
            default: {
                result = cost * (this._amount / 100.0);
                break;
            }
            case PerJob: {
                result = cost + this._amount;
                break;
            }
            case PerPage: {
                result = cost + (totalPages <= 0 ? 0.0 : (double)totalPages * this._amount);
                break;
            }
            case PerCopy: {
                result = cost + (copies <= 0 ? 0.0 : (double)copies * this._amount);
            }
        }
        if (logger.isDebugEnabled() && cost != result) {
            logger.debug("Adjust " + cost + " to " + result + " - " + this.toStringFormat(false));
        }
        return result;
    }

    public static double adjustCost(double cost, List<ChargeRate> ratesIn, int totalPages, int copies) {
        ArrayList<ChargeRate> rates = new ArrayList<ChargeRate>(ratesIn);
        Collections.sort(rates, (r1, r2) -> {
            ChargeRateType t2;
            ChargeRateType t1 = r1.getType();
            if (t1 == (t2 = r2.getType())) {
                return 0;
            }
            if (t1 == ChargeRateType.Percentage) {
                return 1;
            }
            if (t2 == ChargeRateType.Percentage) {
                return -1;
            }
            return 0;
        });
        double result = cost;
        for (ChargeRate rate : rates) {
            result = rate.adjustCost(result, totalPages, copies);
        }
        return result;
    }

    public static ChargeRate parse(String s) {
        if (StringUtils.isNotBlank((String)(s = StringUtils.trimToNull((String)s)))) {
            String[] a = s.split(String.valueOf(':'));
            if (a.length >= 2) {
                double amount;
                ChargeRateType type;
                String desc = a[0];
                String amountStr = StringUtils.trimToEmpty((String)a[1]).toLowerCase();
                if (amountStr.endsWith("%")) {
                    type = ChargeRateType.Percentage;
                    amount = amountStr.startsWith("+") || amountStr.startsWith("-") ? 100.0 + Double.parseDouble(amountStr.replace("%", "")) : Double.parseDouble(amountStr.replace("%", ""));
                } else if (amountStr.endsWith("pp")) {
                    type = ChargeRateType.PerPage;
                    amount = Double.parseDouble(amountStr.replace("pp", ""));
                } else if (amountStr.endsWith("pj")) {
                    type = ChargeRateType.PerJob;
                    amount = Double.parseDouble(amountStr.replace("pj", ""));
                } else if (amountStr.endsWith("pc")) {
                    type = ChargeRateType.PerCopy;
                    amount = Double.parseDouble(amountStr.replace("pc", ""));
                } else {
                    throw new IllegalArgumentException("Unknown charge rate type: " + amountStr);
                }
                Object formattedAmount = a.length > 2 && StringUtils.isNotBlank((String)a[2]) ? StringUtils.trimToEmpty((String)a[2]) : (type == ChargeRateType.Percentage ? ChargeRate.formatPercentForDisplay(amount, null) : (amount >= 0.0 ? "+" : "") + Formatter.formatNumber(amount, 2, null));
                return new ChargeRate(desc, amount, (String)formattedAmount, type);
            }
            throw new IllegalArgumentException("Unexpected ChargeRate text format: " + s);
        }
        return null;
    }

    public String toStringFormat(boolean includeFormattedAmount) {
        switch (this._type) {
            default: {
                return this.getDescription() + ":" + Double.toString(this.getAmount()) + "%" + (String)(includeFormattedAmount ? ":" + this._formattedAmount : "");
            }
            case PerJob: {
                return this.getDescription() + ":" + Double.toString(this.getAmount()) + "pj" + (String)(includeFormattedAmount ? ":" + this._formattedAmount : "");
            }
            case PerCopy: {
                return this.getDescription() + ":" + Double.toString(this.getAmount()) + "pc" + (String)(includeFormattedAmount ? ":" + this._formattedAmount : "");
            }
            case PerPage: 
        }
        return this.getDescription() + ":" + Double.toString(this.getAmount()) + "pp" + (String)(includeFormattedAmount ? ":" + this._formattedAmount : "");
    }

    public String toString() {
        return this.toString(null);
    }

    public String toString(Locale l) {
        switch (this._type) {
            default: {
                String pctStr = ChargeRate.formatPercentForDisplay(this.getAmount(), l);
                return this.getDescription() + " (" + pctStr + ")";
            }
            case PerJob: {
                return this.getDescription() + " (" + Messages.getString(this.getClass(), l, "ChargeRate.PER_JOB", this.getFormattedAmount()) + ")";
            }
            case PerCopy: {
                return this.getDescription() + " (" + Messages.getString(this.getClass(), l, "ChargeRate.PER_COPY", this.getFormattedAmount()) + ")";
            }
            case PerPage: 
        }
        return this.getDescription() + " (" + Messages.getString(this.getClass(), l, "ChargeRate.PER_PAGE", this.getFormattedAmount()) + ")";
    }

    private static String formatPercentForDisplay(double amount, Locale l) {
        double delta = amount - 100.0;
        String sign = delta >= 0.0 ? "+" : "";
        return sign + Formatter.formatNumber(delta, 1, l) + "%";
    }

    public static String listToString(List<ChargeRate> chargeRates, boolean includeFormattedAmount) {
        StringBuilder sb = new StringBuilder();
        for (ChargeRate cr : chargeRates) {
            if (sb.length() > 0) {
                sb.append(',');
            }
            sb.append(cr.toStringFormat(includeFormattedAmount));
        }
        return sb.toString();
    }

    public static List<ChargeRate> stringToList(String s) {
        ArrayList<ChargeRate> l = new ArrayList<ChargeRate>();
        if (StringUtils.isNotBlank((String)s)) {
            String[] a;
            for (String v : a = s.split(String.valueOf(','))) {
                ChargeRate cr = ChargeRate.parse(v);
                if (cr == null) continue;
                l.add(cr);
            }
        }
        return l;
    }

    public static List<ChargeRate> merge(List<ChargeRate> a, List<ChargeRate> b) {
        LinkedHashMap<String, ChargeRate> r = new LinkedHashMap<String, ChargeRate>();
        for (ChargeRate rate : a) {
            r.put(rate.getDescription().trim().toLowerCase(), rate);
        }
        for (ChargeRate rate : b) {
            r.put(rate.getDescription().trim().toLowerCase(), rate);
        }
        return new ArrayList<ChargeRate>(r.values());
    }

    public static enum ChargeRateType {
        Percentage,
        PerPage,
        PerJob,
        PerCopy;

    }
}

